"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("dotenv/config");
const promises_1 = require("node:fs/promises");
const node_path_1 = require("node:path");
const typeorm_1 = require("typeorm");
const speaker_entity_1 = require("./speakers/speaker.entity");
const faq_item_entity_1 = require("./faq/faq-item.entity");
const locale_entry_entity_1 = require("./locales/locale-entry.entity");
const locales_defaults_1 = require("./locales/locales.defaults");
const rootDir = process.cwd();
const DEFAULT_DB_URL = "postgresql://offered_user:1231230@localhost:5432/offered";
const dataSource = new typeorm_1.DataSource({
    type: "postgres",
    url: process.env.DATABASE_URL ?? DEFAULT_DB_URL,
    entities: [speaker_entity_1.Speaker, faq_item_entity_1.FaqItem, locale_entry_entity_1.LocaleEntry],
    synchronize: true,
});
async function readJson(path) {
    const content = await (0, promises_1.readFile)(path, "utf-8");
    return JSON.parse(content);
}
function flattenLocale(obj, prefix = "", out = {}) {
    for (const key of Object.keys(obj)) {
        const value = obj[key];
        const path = prefix ? `${prefix}.${key}` : key;
        if (value && typeof value === "object" && !Array.isArray(value)) {
            flattenLocale(value, path, out);
        }
        else {
            out[path] = value === undefined || value === null ? "" : String(value);
        }
    }
    return out;
}
async function loadLocaleEntries() {
    const localesDir = process.env.LOCALES_DIR ??
        (0, node_path_1.resolve)(rootDir, "../front/src/locales");
    const entries = [];
    try {
        const files = await (0, promises_1.readdir)(localesDir);
        const localeFiles = files.filter((file) => (0, node_path_1.extname)(file) === ".json");
        for (const file of localeFiles) {
            const locale = file.replace(/\.json$/i, "");
            const data = await readJson((0, node_path_1.resolve)(localesDir, file));
            const flat = flattenLocale(data);
            for (const [key, value] of Object.entries(flat)) {
                entries.push({ locale, key, value });
            }
        }
    }
    catch (error) {
        console.warn(`Locales directory not found (${localesDir}), only defaults will be seeded.`);
    }
    for (const entry of locales_defaults_1.DEFAULT_LOCALE_ENTRIES) {
        entries.push(entry);
    }
    return entries;
}
async function seedLocales() {
    const localeRepo = dataSource.getRepository(locale_entry_entity_1.LocaleEntry);
    const entries = await loadLocaleEntries();
    if (entries.length === 0) {
        console.warn("No locale entries found, skipping locale seed.");
        return;
    }
    await localeRepo.upsert(entries, ["locale", "key"]);
    console.log(`Locales seeded: ${entries.length} entries`);
}
async function seedSpeakers() {
    const speakersPath = process.env.SPEAKERS_SEED_PATH;
    if (!speakersPath) {
        console.log("SPEAKERS_SEED_PATH not set, skipping speakers seed.");
        return;
    }
    const speakerRepo = dataSource.getRepository(speaker_entity_1.Speaker);
    const force = process.env.SEED_FORCE === "1";
    const existing = await speakerRepo.count();
    if (existing > 0 && !force) {
        console.log("Speakers already exist, skipping. Set SEED_FORCE=1 to overwrite.");
        return;
    }
    const source = await readJson((0, node_path_1.resolve)(rootDir, speakersPath));
    if (force && existing > 0) {
        await speakerRepo.clear();
    }
    const toInsert = source.map((speaker, index) => speakerRepo.create({
        ...speaker,
        order: speaker.order ?? index,
    }));
    await speakerRepo.save(toInsert);
    console.log(`Speakers seeded: ${toInsert.length} items`);
}
async function seedFaq() {
    const faqPath = process.env.FAQ_SEED_PATH;
    if (!faqPath) {
        console.log("FAQ_SEED_PATH not set, skipping FAQ seed.");
        return;
    }
    const faqRepo = dataSource.getRepository(faq_item_entity_1.FaqItem);
    const force = process.env.SEED_FORCE === "1";
    const existing = await faqRepo.count();
    if (existing > 0 && !force) {
        console.log("FAQ items already exist, skipping. Set SEED_FORCE=1 to overwrite.");
        return;
    }
    const source = await readJson((0, node_path_1.resolve)(rootDir, faqPath));
    if (force && existing > 0) {
        await faqRepo.clear();
    }
    const toInsert = source.map((item, index) => faqRepo.create({
        ...item,
        order: item.order ?? index,
    }));
    await faqRepo.save(toInsert);
    console.log(`FAQ seeded: ${toInsert.length} items`);
}
async function seed() {
    await dataSource.initialize();
    try {
        await seedLocales();
        await seedSpeakers();
        await seedFaq();
    }
    finally {
        await dataSource.destroy();
    }
}
seed().catch((error) => {
    console.error("Seed failed", error);
    process.exitCode = 1;
});
//# sourceMappingURL=seed.js.map